'use strict';

describe('services/userService.js', function () {

  var $user, $q, $scope, $session, $rootScope, $location, $api = null;

  beforeEach(module('genisis'));

  beforeEach(function () {
    inject(function (user, _$q_, _$rootScope_, _$location_,
      $httpBackend, api,
      session
    ) {
      $user = user;
      $q = _$q_;
      $session = session;
      $api = api;
      $location = _$location_;
      $rootScope = _$rootScope_;
      $rootScope.errors = [];
      $scope = $rootScope.$new();

      $httpBackend.when('GET').respond('test GET response');
      $session.delete();
    });
  });


  it('service exists', function () {
    expect($user).not.toBeNull();
  });

  describe('$user.info() function', function () {

    it('exists', function () {
      expect($user.info).toBeDefined();
    });

    it('returns an object', function () {
      expect($user.info()).toEqual({});
    });
  });

  describe('$user.login() function', function () {
    it('exists', function () {
      expect($user.login).toBeDefined();
    });

    it('accepts a valid login', function () {
      var fakeUser = {
        userId: 1,
        emailId: 1,
        username: 'test',
        status: 'Active',
        first: 'test',
        last: 'tester'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      spyOn($location, 'path').and.returnValue('/');

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($location.path.calls.count()).toEqual(1);
        expect($location.path.calls.first().args).toEqual([
          '/requests'
        ]);
        expect($user.info()).toEqual(fakeUser);
      });

      $scope.$apply();
    });

    it('returns API error messages', function () {

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: false,
          response: null,
          message: 'some API error'
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(false);
        expect($rootScope.errors).toEqual(['some API error']);
      });

      $scope.$apply();
    });

    it('returns an error message if there is no API response', function () {

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve(null);
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(false);
        expect($user.info()).toEqual({});
        expect($rootScope.errors).toEqual([
          'Invalid API Response'
        ]);
      });

      $scope.$apply();
    });
  });

  describe('$user.logout() function', function () {
    it('exists', function () {
      expect($user.logout).toBeDefined();
    });

    it('removes the user\'s session data', function () {
      var fakeUser = {
        userId: 1,
        emailId: 1,
        first: 'test',
        status: 'Active',
        last: 'tester'
      };

      $session.set('user', fakeUser);
      $user.logout();
      expect($user.info()).toEqual({});
      expect($session.get()).toEqual({});
    });
  });

  describe('$user.id() function', function () {
    it('exists', function () {
      expect($user.id).toBeDefined();
    });

    it('returns the correct user id', function () {
      var fakeUser = {
        userId: 2,
        emailId: 2,
        first: 'test',
        status: 'Active',
        last: 'tester'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($user.info()).toEqual(fakeUser);
        expect($user.id()).toEqual(fakeUser.userId);
      });

      $user.login('fakeEmail');
      $scope.$apply();
    });
  });

  describe('$user.name() function', function () {
    it('exists', function () {
      expect($user.name).toBeDefined();
    });

    it('returns the correct user id', function () {
      var fakeUser = {
        userId: 2,
        emailId: 2,
        firstName: 'test',
        status: 'Active',
        lastName: 'tester'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($user.info()).toEqual(fakeUser);
        expect($user.name()).toEqual(
          fakeUser.firstName + ' ' + fakeUser.lastName
        );
      });

      $user.login('fakeEmail');
      $scope.$apply();
    });
  });

  describe('$user.loggedIn() function', function () {

    it('exists', function () {
      expect($user.loggedIn).toBeDefined();
    });

    it('returns true when the user has an id', function () {
      $user.logout();

      var fakeUser = {
        userId: 2,
        emailId: 2,
        status: 'Active',
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($user.info()).toEqual(fakeUser);
        expect($user.loggedIn()).toEqual(true);
      });

      $user.login('fakeEmail');
      $scope.$apply();
    });

    it('returns false when there is no user id', function () {
      $user.logout();
      expect($user.loggedIn()).toEqual(false);
    });
  });

  describe('$user.permissions() function', function () {
    it('exists', function () {
      expect($user.permissions).toBeDefined();
    });

    it('returns the user\'s permissions', function () {
      var fakeUser = {
        userId: 2,
        emailId: 2,
        firstName: 'test',
        lastName: 'tester',
        status: 'Active',
        role: 'researcher'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($user.info()).toEqual(fakeUser);
        expect($user.permissions()).toEqual(fakeUser.role);
      });

      $user.login('fakeEmail');
      $scope.$apply();
    });
  });

  describe('$user.requester() function', function () {
    it('exists', function () {
      expect($user.requester).toBeDefined();
    });

    it('returns true if the user has a requester role', function () {
      var fakeUser = {
        userId: 2,
        emailId: 2,
        firstName: 'test',
        lastName: 'tester',
        status: 'Active',
        role: 'requester'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($user.info()).toEqual(fakeUser);
        expect($user.requester()).toEqual(true);
      });

      $user.login('fakeEmail');
      $scope.$apply();
    });

    it('returns true if the user has a requester role', function () {
      $user.logout();

      var fakeUser = {
        userId: 2,
        emailId: 2,
        firstName: 'test',
        lastName: 'tester',
        status: 'Active',
        role: 'something else'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($user.info()).toEqual(fakeUser);
        expect($user.requester()).toEqual(false);
      });

      $user.login('fakeEmail');
      $scope.$apply();
    });
  });

  describe('$user.genisis2Mgr() function', function () {
    it('exists', function () {
      expect($user.genisis2Mgr).toBeDefined();
    });

    it('returns true if the user has a genisis2Mgr role', function () {
      var fakeUser = {
        userId: 2,
        emailId: 2,
        firstName: 'test',
        lastName: 'tester',
        status: 'Active',
        role: 'Genisis2 Manager'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($user.info()).toEqual(fakeUser);
        expect($user.genisis2Mgr()).toEqual(true);
      });

      $user.login('fakeEmail');
      $scope.$apply();
    });

    it('returns false if the user does not have a manager role',
      function () {
        $user.logout();

        var fakeUser = {
          userId: 2,
          emailId: 2,
          firstName: 'test',
          lastName: 'tester',
          status: 'Active',
          role: 'something else'
        };

        spyOn($api, 'fetch').and.callFake(function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: fakeUser
          });
          return deferred.promise;
        });

        var promise = $user.login('fakeEmail');

        promise.then(function (data) {
          expect(data).toEqual(true);
          expect($user.info()).toEqual(fakeUser);
          expect($user.genisis2Mgr()).toEqual(false);
        });

        $user.login('fakeEmail');
        $scope.$apply();
      });
  });

  describe('$user.vinciMgr() function', function () {
    it('exists', function () {
      expect($user.vinciMgr).toBeDefined();
    });

    it('returns true if the user has a vinciMgr role', function () {
      var fakeUser = {
        userId: 2,
        emailId: 2,
        firstName: 'test',
        lastName: 'tester',
        status: 'Active',
        role: 'VINCI Manager'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($user.info()).toEqual(fakeUser);
        expect($user.vinciMgr()).toEqual(true);
      });

      $user.login('fakeEmail');
      $scope.$apply();
    });

    it('returns false if the user does not have a source role',
      function () {
        $user.logout();

        var fakeUser = {
          userId: 2,
          emailId: 2,
          firstName: 'test',
          lastName: 'tester',
          status: 'Active',
          role: 'something else'
        };

        spyOn($api, 'fetch').and.callFake(function () {
          var deferred = $q.defer();
          deferred.resolve({
            success: true,
            response: fakeUser
          });
          return deferred.promise;
        });

        var promise = $user.login('fakeEmail');

        promise.then(function (data) {
          expect(data).toEqual(true);
          expect($user.info()).toEqual(fakeUser);
          expect($user.vinciMgr()).toEqual(false);
        });

        $user.login('fakeEmail');
        $scope.$apply();
      });
  });

  describe('$user.Administrator() function', function () {
    it('exists', function () {
      expect($user.Administrator).toBeDefined();
    });

    it('returns true if the user has a researcher role', function () {
      var fakeUser = {
        userId: 2,
        emailId: 2,
        firstName: 'test',
        lastName: 'tester',
        status: 'Active',
        role: 'Administrator'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($user.info()).toEqual(fakeUser);
        expect($user.Administrator()).toEqual(true);
      });

      $user.login('fakeEmail');
      $scope.$apply();
    });

    it('returns true if the user has an admin role', function () {
      $user.logout();

      var fakeUser = {
        userId: 2,
        emailId: 2,
        firstName: 'test',
        lastName: 'tester',
        status: 'Active',
        role: 'something else'
      };

      spyOn($api, 'fetch').and.callFake(function () {
        var deferred = $q.defer();
        deferred.resolve({
          success: true,
          response: fakeUser
        });
        return deferred.promise;
      });

      var promise = $user.login('fakeEmail');

      promise.then(function (data) {
        expect(data).toEqual(true);
        expect($user.info()).toEqual(fakeUser);
        expect($user.Administrator()).toEqual(false);
      });

      $user.login('fakeEmail');
      $scope.$apply();
    });
  });
});